<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Organization;
use App\Services\MetaApiService;
use Illuminate\Http\Request;
use Illuminate\Http\Client\RequestException;
use Inertia\Inertia;

class OrganizationsController extends Controller
{
    public function index()
    {
        if (!auth()->user()->isSuperAdmin()) {
            abort(403, 'UNAUTHORIZED ACCESS');
        }

        $organizations = Organization::query()
            ->withCount('users')
            ->orderBy('created_at', 'desc')
            ->get()
            ->map(function (Organization $org) {
                $hasToken = !empty($org->getRawOriginal('system_user_token'));
                $expiresAt = $org->system_user_token_expires_at;
                $requiresPermanentUpgrade = (bool) data_get($org->settings, 'requires_permanent_token_upgrade', false);

                $isTokenValid = $hasToken && (!$expiresAt || $expiresAt->isFuture());
                $isHealthy = !empty($org->waba_id) && !empty($org->phone_number_id) && $isTokenValid;

                return [
                    'id' => $org->id,
                    'name' => $org->name,
                    'users_count' => $org->users_count,
                    'waba_id' => $org->waba_id,
                    'phone_number_id' => $org->phone_number_id,
                    'token_expires_at' => $expiresAt?->toISOString(),
                    'connection_status' => $isHealthy ? 'healthy' : 'unhealthy',
                    'requires_permanent_token_upgrade' => $requiresPermanentUpgrade,
                    'created_at' => $org->created_at?->toISOString(),
                ];
            })
            ->values();

        return Inertia::render('Admin/Organizations', [
            'organizations' => $organizations,
        ]);
    }

    public function diagnostic(Request $request, Organization $organization, MetaApiService $metaService)
    {
        if (!auth()->user()->isSuperAdmin()) {
            abort(403, 'UNAUTHORIZED ACCESS');
        }

        try {
            $metaService->setOrganization($organization);

            if (!empty($organization->phone_number_id)) {
                $data = $metaService->get('/' . $organization->phone_number_id, [
                    'fields' => 'id,display_phone_number,verified_name,quality_rating',
                ]);
            } elseif (!empty($organization->waba_id)) {
                $data = $metaService->get('/' . $organization->waba_id, [
                    'fields' => 'id,name',
                ]);
            } else {
                $data = $metaService->get('/me', [
                    'fields' => 'id',
                ]);
            }

            return response()->json([
                'success' => true,
                'data' => $data,
            ]);
        } catch (RequestException $e) {
            return response()->json([
                'success' => false,
                'status' => $e->response?->status(),
                'error' => $e->response?->json(),
            ]);
        } catch (\Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
            ]);
        }
    }
}
