<?php

namespace App\Http\Controllers;

use App\Jobs\ProcessCampaign;
use App\Models\Campaign;
use App\Models\Contact;
use Illuminate\Http\Request;
use Inertia\Inertia;

class CampaignController extends Controller
{
    /**
     * Display a listing of the campaigns.
     */
    public function index()
    {
        $organization = auth()->user()->organization;

        $campaigns = Campaign::where('organization_id', $organization->id)
            ->withCount('contacts')
            ->orderBy('created_at', 'desc')
            ->get();

        return Inertia::render('Campaigns/Index', [
            'organization' => $organization,
            'campaigns' => $campaigns,
        ]);
    }

    /**
     * Show the form for creating a new campaign.
     */
    public function create()
    {
        $organization = auth()->user()->organization;

        return Inertia::render('Campaigns/Create', [
            'organization' => $organization,
        ]);
    }

    /**
     * Store a newly created campaign in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'template_name' => 'required|string|max:255',
            'language' => 'required|string|max:10',
            'scheduled_at' => 'nullable|date|after:now',
            // In a real app, you would validate components structure
        ]);

        $organization = auth()->user()->organization;

        $campaign = Campaign::create([
            'organization_id' => $organization->id,
            'name' => $request->name,
            'template_name' => $request->template_name,
            'language' => $request->language,
            'scheduled_at' => $request->scheduled_at,
            'status' => $request->scheduled_at ? 'scheduled' : 'draft',
        ]);

        // Attach all opted-in contacts to the campaign
        $contactIds = Contact::where('organization_id', $organization->id)
            ->where('opted_out', false)
            ->pluck('id');
            
        $campaign->contacts()->attach($contactIds);
        
        $campaign->update(['total_recipients' => $contactIds->count()]);

        // If not scheduled for later, process immediately
        if (!$request->scheduled_at) {
            $campaign->update(['status' => 'scheduled']);
            ProcessCampaign::dispatch($campaign->id);
        }

        return redirect()->route('campaigns.index')->with('success', 'Campaign created successfully.');
    }
}
