<?php

namespace App\Http\Controllers;

use App\Models\CustomFieldDefinition;
use Illuminate\Http\Request;

class CustomFieldController extends Controller
{
    public function index(Request $request)
    {
        $fields = CustomFieldDefinition::where('organization_id', auth()->user()->organization_id)
            ->ordered()
            ->get();

        return response()->json($fields);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'key' => 'required|string|max:190|unique:custom_field_definitions,key,NULL,id,organization_id,' . auth()->user()->organization_id,
            'label' => 'required|string|max:190',
            'type' => 'required|in:text,textarea,number,date,select,checkbox,email,url',
            'required' => 'boolean',
            'default_visible' => 'boolean',
            'options' => 'nullable|array',
            'options.*' => 'string|max:190',
            'default_value' => 'nullable|string|max:190',
        ]);

        $validated['organization_id'] = auth()->user()->organization_id;
        $validated['sort_order'] = CustomFieldDefinition::where('organization_id', auth()->user()->organization_id)->max('sort_order') + 1;

        $field = CustomFieldDefinition::create($validated);

        return response()->json($field);
    }

    public function update(Request $request, CustomFieldDefinition $field)
    {
        abort_if($field->organization_id !== auth()->user()->organization_id, 403);

        $validated = $request->validate([
            'label' => 'required|string|max:190',
            'type' => 'required|in:text,textarea,number,date,select,checkbox,email,url',
            'required' => 'boolean',
            'default_visible' => 'boolean',
            'options' => 'nullable|array',
            'options.*' => 'string|max:190',
            'default_value' => 'nullable|string|max:190',
        ]);

        $field->update($validated);

        return response()->json($field);
    }

    public function destroy(CustomFieldDefinition $field)
    {
        abort_if($field->organization_id !== auth()->user()->organization_id, 403);

        $field->delete();

        return response()->json(['message' => 'Custom field deleted successfully']);
    }

    public function reorder(Request $request)
    {
        $validated = $request->validate([
            'fields' => 'required|array',
            'fields.*.id' => 'required|exists:custom_field_definitions,id',
            'fields.*.sort_order' => 'required|integer|min:0',
        ]);

        foreach ($validated['fields'] as $fieldData) {
            $field = CustomFieldDefinition::find($fieldData['id']);
            if ($field && $field->organization_id === auth()->user()->organization_id) {
                $field->update(['sort_order' => $fieldData['sort_order']]);
            }
        }

        return response()->json(['message' => 'Fields reordered successfully']);
    }
}
