<?php

namespace App\Http\Controllers;

use App\Models\KnowledgeBaseDocument;
use Illuminate\Http\Request;
use Inertia\Inertia;

class KnowledgeBaseController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $organization = auth()->user()->organization;

        $documents = KnowledgeBaseDocument::where('organization_id', $organization->id)
            ->orderBy('created_at', 'desc')
            ->get();

        return Inertia::render('KnowledgeBase/Index', [
            'organization' => $organization,
            'documents' => $documents,
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'content' => 'required|string',
            'type' => 'required|string|in:text,faq,policy',
            'is_active' => 'boolean',
        ]);

        $organization = auth()->user()->organization;

        KnowledgeBaseDocument::create([
            'organization_id' => $organization->id,
            'title' => $request->title,
            'content' => $request->content,
            'type' => $request->type,
            'is_active' => $request->is_active ?? true,
        ]);

        return back()->with('success', 'Document created successfully.');
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, KnowledgeBaseDocument $knowledgeBase)
    {
        if ($knowledgeBase->organization_id !== auth()->user()->organization_id) {
            abort(403);
        }

        $validated = $request->validate([
            'title' => 'sometimes|required|string|max:255',
            'content' => 'sometimes|required|string',
            'type' => 'sometimes|required|string|in:text,faq,policy',
            'is_active' => 'boolean',
        ]);

        $knowledgeBase->update($validated);

        return back()->with('success', 'Document updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(KnowledgeBaseDocument $knowledgeBase)
    {
        if ($knowledgeBase->organization_id !== auth()->user()->organization_id) {
            abort(403);
        }

        $knowledgeBase->delete();

        return back()->with('success', 'Document deleted successfully.');
    }
}
