<?php

namespace App\Http\Controllers;

use App\Models\Organization;
use App\Models\User;
use App\Services\MetaApiService;
use App\Services\PlatformSettingsService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Inertia\Inertia;
use Illuminate\Validation\Rules\Password;
use Illuminate\Support\Carbon;

class OrganizationOnboardingController extends Controller
{
    /**
     * Show the registration form.
     */
    public function showRegistrationForm()
    {
        return Inertia::render('Auth/Register');
    }

    /**
     * Handle the registration request.
     */
    public function register(Request $request)
    {
        $request->validate([
            'organization_name' => ['required', 'string', 'max:255'],
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users'],
            'password' => ['required', 'confirmed', Password::defaults()],
        ]);

        DB::transaction(function () use ($request) {
            $organization = Organization::create([
                'name' => $request->organization_name,
            ]);

            $user = User::create([
                'organization_id' => $organization->id,
                'name' => $request->name,
                'email' => $request->email,
                'password' => Hash::make($request->password),
                'role' => 'owner',
            ]);

            Auth::login($user);
        });

        return redirect()->route('dashboard');
    }

    /**
     * Handle the Meta Embedded Signup callback.
     * Exchanges the code for a token and retrieves WABA details.
     */
    public function handleMetaCallback(Request $request, MetaApiService $metaService, PlatformSettingsService $platformSettings)
    {
        if (!Auth::user()->hasRole(['owner', 'system_admin'])) {
            abort(403);
        }

        $validated = $request->validate([
            'code' => ['nullable', 'string', 'required_without:access_token'],
            'access_token' => ['nullable', 'string', 'required_without:code'],
            'waba_id' => ['nullable', 'string'],
            'phone_number_id' => ['nullable', 'string'],
        ]);

        $user = Auth::user();
        $organization = $user->organization;

        if (!$organization) {
            return back()->with('error', 'Organization context is missing for this account.');
        }

        try {
            $accessToken = $validated['access_token'] ?? null;

            $metaAppId = $platformSettings->get('meta.app_id', config('meta.app_id'), false);
            $metaAppSecret = $platformSettings->get('meta.app_secret', config('meta.app_secret'));
            $metaBusinessId = $platformSettings->get('meta.business_id', config('meta.business_id'), false);
            $metaRedirectUri = $platformSettings->get('meta.redirect_uri', config('meta.redirect_uri'), false);

            // If a short-lived OAuth code was provided, exchange it for an access token.
            if (!$accessToken && !empty($validated['code'])) {
                $tokenResponse = $metaService->publicGet('/oauth/access_token', [
                    'client_id' => $metaAppId,
                    'client_secret' => $metaAppSecret,
                    'code' => $validated['code'],
                    'redirect_uri' => $metaRedirectUri,
                ]);

                $accessToken = $tokenResponse['access_token'] ?? null;

                if (!$accessToken) {
                    throw new \Exception('Meta did not return an access token.');
                }
            }

            $expiresAt = null;

            if ($accessToken) {
                try {
                    $longLivedResponse = $metaService->publicGet('/oauth/access_token', [
                        'grant_type' => 'fb_exchange_token',
                        'client_id' => $metaAppId,
                        'client_secret' => $metaAppSecret,
                        'fb_exchange_token' => $accessToken,
                    ]);

                    $longLivedToken = $longLivedResponse['access_token'] ?? null;
                    $expiresIn = $longLivedResponse['expires_in'] ?? null;

                    if ($longLivedToken) {
                        $accessToken = $longLivedToken;
                    }

                    if (is_numeric($expiresIn) && (int) $expiresIn > 0) {
                        $expiresAt = Carbon::now()->addSeconds((int) $expiresIn);
                    }
                } catch (\Throwable $e) {
                }
            }

            try {
                $debug = $metaService->publicGet('/debug_token', [
                    'input_token' => $accessToken,
                    'access_token' => $metaAppId . '|' . $metaAppSecret,
                ]);

                $debugData = $debug['data'] ?? [];

                if (array_key_exists('is_valid', $debugData) && $debugData['is_valid'] === false) {
                    throw new \Exception('Meta returned an invalid token during verification.');
                }

                $expiresAtUnix = $debugData['expires_at'] ?? null;

                if (is_numeric($expiresAtUnix) && (int) $expiresAtUnix > 0) {
                    $expiresAt = Carbon::createFromTimestamp((int) $expiresAtUnix);
                }

                if (is_numeric($expiresAtUnix) && (int) $expiresAtUnix === 0) {
                    $expiresAt = null;
                }
            } catch (\Throwable $e) {
            }

            // Persist token (encrypted via cast)
            $organization->update([
                'system_user_token' => $accessToken,
                'system_user_token_expires_at' => $expiresAt,
            ]);

            $metaService->setOrganization($organization);

            $wabaId = $validated['waba_id'] ?? null;
            $phoneNumberId = $validated['phone_number_id'] ?? null;

            if (!$wabaId) {
                // Try to retrieve WABA from the configured business context first.
                if ($metaBusinessId) {
                    try {
                        $wabaResponse = $metaService->get('/' . $metaBusinessId . '/client_whatsapp_business_accounts', [
                            'fields' => 'id,name',
                        ]);
                        $wabaId = $wabaResponse['data'][0]['id'] ?? null;
                    } catch (\Throwable $e) {
                        // Fall through to other discovery endpoints.
                    }
                }

                // Fallback: try the current user context.
                if (!$wabaId) {
                    try {
                        $wabaResponse = $metaService->get('/me/whatsapp_business_accounts', [
                            'fields' => 'id,name',
                        ]);
                        $wabaId = $wabaResponse['data'][0]['id'] ?? null;
                    } catch (\Throwable $e) {
                        // Ignore discovery failure; token may still be valid.
                    }
                }
            }

            if ($wabaId && !$phoneNumberId) {
                try {
                    $phoneResponse = $metaService->get('/' . $wabaId . '/phone_numbers', [
                        'fields' => 'id,display_phone_number,verified_name',
                    ]);
                    $phoneNumberId = $phoneResponse['data'][0]['id'] ?? null;
                } catch (\Throwable $e) {
                    // Ignore; user may not have provisioned a phone yet.
                }
            }

            $organization->update([
                'waba_id' => $wabaId,
                'phone_number_id' => $phoneNumberId,
            ]);

            $requiresPermanentTokenUpgrade = false;

            if ($accessToken && $expiresAt) {
                $existingSettings = $organization->settings ?? [];
                $storedSystemUserId = data_get($existingSettings, 'meta_system_user_id');
                $businessId = null;

                try {
                    $businessResponse = $metaService->tokenGet($accessToken, '/me/businesses', [
                        'fields' => 'id,name',
                    ]);
                    $businessId = $businessResponse['data'][0]['id'] ?? null;
                } catch (\Throwable $e) {
                    $businessId = null;
                }

                $systemUserId = null;
                if (!empty($storedSystemUserId)) {
                    $systemUserId = $storedSystemUserId;
                }
                if ($businessId) {
                    try {
                        $users = $metaService->tokenGet($accessToken, '/' . $businessId . '/system_users', [
                            'fields' => 'id,name,role',
                        ]);
                        if (!$systemUserId) {
                            $systemUserId = $users['data'][0]['id'] ?? null;
                        }
                    } catch (\Throwable $e) {
                    }
                }

                if (!$systemUserId && $businessId) {
                    try {
                        $createdUser = $metaService->tokenPost($accessToken, '/' . $businessId . '/system_users', [
                            'name' => 'WABA Manager - Org ' . $organization->id,
                            'role' => 'ADMIN',
                        ]);
                        $systemUserId = $createdUser['id'] ?? null;
                    } catch (\Throwable $e) {
                        $systemUserId = null;
                    }
                }

                $permanentToken = null;
                $scopes = 'whatsapp_business_management,whatsapp_business_messaging';

                if ($systemUserId) {
                    try {
                        $tokenResponse = $metaService->tokenPost($accessToken, '/' . $systemUserId . '/access_tokens', [
                            'app_id' => $metaAppId,
                            'scope' => $scopes,
                        ]);
                        $permanentToken = $tokenResponse['access_token'] ?? null;
                    } catch (\Throwable $e) {
                        $permanentToken = null;
                    }
                }

                if ($permanentToken) {
                    $mintedExpiresAt = null;
                    $mintedRequiresUpgrade = false;

                    try {
                        $debug = $metaService->publicGet('/debug_token', [
                            'input_token' => $permanentToken,
                            'access_token' => $metaAppId . '|' . $metaAppSecret,
                        ]);

                        $debugData = $debug['data'] ?? [];
                        $expiresAtUnix = $debugData['expires_at'] ?? null;

                        if (is_numeric($expiresAtUnix) && (int) $expiresAtUnix > 0) {
                            $mintedExpiresAt = Carbon::createFromTimestamp((int) $expiresAtUnix);
                            $mintedRequiresUpgrade = true;
                        }
                    } catch (\Throwable $e) {
                        $mintedRequiresUpgrade = true;
                    }

                    $settings = $organization->settings ?? [];
                    $settings['requires_permanent_token_upgrade'] = $mintedRequiresUpgrade;
                    if ($businessId) {
                        $settings['meta_business_id'] = $businessId;
                    }
                    if ($systemUserId) {
                        $settings['meta_system_user_id'] = $systemUserId;
                    }

                    $organization->update([
                        'system_user_token' => $permanentToken,
                        'system_user_token_expires_at' => $mintedExpiresAt,
                        'settings' => $settings,
                    ]);
                } else {
                    $requiresPermanentTokenUpgrade = true;

                    $settings = $organization->settings ?? [];
                    $settings['requires_permanent_token_upgrade'] = true;
                    if ($businessId) {
                        $settings['meta_business_id'] = $businessId;
                    }
                    $organization->update([
                        'settings' => $settings,
                    ]);
                }
            }

            if (!$expiresAt) {
                $settings = $organization->settings ?? [];
                $settings['requires_permanent_token_upgrade'] = false;
                $organization->update([
                    'settings' => $settings,
                ]);
            }

            return back()->with('success', 'WhatsApp connected successfully.');
        } catch (\Throwable $e) {
            return back()->with('error', $e->getMessage());
        }
    }

    public function disconnectMeta(Request $request)
    {
        if (!Auth::user()->hasRole(['owner', 'system_admin'])) {
            abort(403);
        }

        $organization = Auth::user()->organization;

        $organization->update([
            'waba_id' => null,
            'phone_number_id' => null,
            'system_user_token' => null,
            'system_user_token_expires_at' => null,
        ]);

        $settings = $organization->settings ?? [];
        $settings['requires_permanent_token_upgrade'] = false;
        $organization->update([
            'settings' => $settings,
        ]);

        return back()->with('success', 'WhatsApp disconnected successfully.');
    }
}
