<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\OrganizationRole;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Inertia\Inertia;
use Illuminate\Validation\Rule;

class TeamController extends Controller
{
    /**
     * Display the team management dashboard for the organization.
     */
    public function index()
    {
        $organization = auth()->user()->organization;

        // Ensure only Tenant Owners or System Admins can access this page
        if (auth()->user()->role !== "Owner" && auth()->user()->role !== "System_Admin") {
            abort(403, "Unauthorized action. Only Owners can manage the team.");
        }

        $team = User::where("organization_id", $organization->id)
            ->with("organizationRole")
            ->orderBy("created_at", "desc")
            ->get(["id", "name", "email", "role", "organization_role_id", "status", "created_at"]);

        $roles = OrganizationRole::where("organization_id", $organization->id)
            ->orderBy("name")
            ->get();

        return Inertia::render("Dashboard/Team", [
            "team" => $team,
            "roles" => $roles,
        ]);
    }

    /**
     * Store a newly created team member in the organization.
     */
    public function store(Request $request)
    {
        // Ensure only Tenant Owners or System Admins can create team members
        if (auth()->user()->role !== "Owner" && auth()->user()->role !== "System_Admin") {
            abort(403, "Unauthorized action. Only Owners can manage the team.");
        }

        $organization = auth()->user()->organization;

        $validated = $request->validate([
            "name" => ["required", "string", "max:255"],
            "email" => ["required", "string", "email", "max:255", "unique:users"],
            "password" => ["required", "string", "min:8", "confirmed"],
            "role" => ["required", "string", Rule::in(["Agent", "Owner"])],
            "status" => ["nullable", "string", Rule::in(["Active", "Inactive"])],
            "organization_role_id" => [
                "nullable",
                Rule::exists("organization_roles", "id")->where(function ($query) use ($organization) {
                    $query->where("organization_id", $organization->id);
                }),
            ],
        ]);

        User::create([
            "organization_id" => $organization->id,
            "name" => $validated["name"],
            "email" => $validated["email"],
            "password" => Hash::make($validated["password"]),
            "role" => $validated["role"],
            "status" => $validated["status"] ?? "Active",
            "organization_role_id" => $validated["organization_role_id"] ?? null,
        ]);

        return back()->with("success", "Team member invited successfully.");
    }

    /**
     * Update the specified team member in storage.
     */
    public function update(Request $request, User $user)
    {
        // Ensure only Tenant Owners or System Admins can update team members
        if (auth()->user()->role !== "Owner" && auth()->user()->role !== "System_Admin") {
            abort(403, "Unauthorized action. Only Owners can manage the team.");
        }

        // Ensure the user belongs to the same organization
        if ($user->organization_id !== auth()->user()->organization_id) {
            abort(403, "Unauthorized action.");
        }

        $validated = $request->validate([
            "name" => ["required", "string", "max:255"],
            "email" => ["required", "string", "email", "max:255", Rule::unique("users")->ignore($user->id)],
            "role" => ["required", "string", Rule::in(["Agent", "Owner"])],
            "status" => ["nullable", "string", Rule::in(["Active", "Inactive"])],
            "organization_role_id" => [
                "nullable",
                Rule::exists("organization_roles", "id")->where(function ($query) use ($user) {
                    $query->where("organization_id", $user->organization_id);
                }),
            ],
            "password" => ["nullable", "string", "min:8", "confirmed"],
        ]);

        $updateData = [
            "name" => $validated["name"],
            "email" => $validated["email"],
            "role" => $validated["role"],
            "status" => $validated["status"] ?? "Active",
            "organization_role_id" => $validated["organization_role_id"] ?? null,
        ];

        if (!empty($validated["password"])) {
            $updateData["password"] = Hash::make($validated["password"]);
        }

        $user->update($updateData);

        return back()->with("success", "Team member updated successfully.");
    }

    /**
     * Remove the specified team member from storage.
     */
    public function destroy(User $user)
    {
        // Ensure only Tenant Owners or System Admins can delete team members
        if (auth()->user()->role !== "Owner" && auth()->user()->role !== "System_Admin") {
            abort(403, "Unauthorized action. Only Owners can manage the team.");
        }

        // Ensure the user belongs to the same organization
        if ($user->organization_id !== auth()->user()->organization_id) {
            abort(403, "Unauthorized action.");
        }

        // Prevent deleting the last owner or deleting oneself
        if ($user->id === auth()->id()) {
            return back()->withErrors(["email" => "You cannot delete your own account from here."]);
        }

        $user->delete();

        return back()->with("success", "Team member deleted successfully.");
    }
}
