<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use App\Models\Message;

class Contact extends BaseTenantModel
{
    use HasFactory;

    /**
     * The "booted" method of the model.
     */
    protected static function booted(): void
    {
        parent::booted();

        static::saving(function (Contact $contact): void {
            // Mirror opt-out flags so automation + UI stay consistent.
            $optedOutDirty = $contact->isDirty('opted_out');
            $isOptedOutDirty = $contact->isDirty('is_opted_out');

            if ($optedOutDirty && ! $isOptedOutDirty) {
                $contact->is_opted_out = (bool) $contact->opted_out;
            } elseif ($isOptedOutDirty && ! $optedOutDirty) {
                $contact->opted_out = (bool) $contact->is_opted_out;
            } elseif (! $optedOutDirty && ! $isOptedOutDirty) {
                // Ensure both fields share the same value even if neither was explicitly changed this request.
                $contact->is_opted_out = (bool) $contact->is_opted_out;
                $contact->opted_out = $contact->is_opted_out;
            }

            // Keep status aligned with opt-out flags.
            if ($contact->opted_out) {
                $contact->status = 'Opted-out';
            } elseif ($contact->status === null || $contact->status === 'Opted-out') {
                $contact->status = 'Active';
            }

            // Backfill phone_number if blank so outbound sends have a target.
            if (! $contact->phone_number && $contact->phone) {
                $contact->phone_number = $contact->phone;
            }
        });
    }

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        'organization_id',
        'first_name',
        'last_name',
        'phone',
        'phone_number',
        'email',
        'dob',
        'gender',
        'address',
        'city',
        'country',
        'po_box',
        'timezone',
        'is_opted_out',
        'opted_out',
        'custom_values',
        'last_messaged_at',
        'last_marketed_at',
        'status',
        'assigned_to',
        'conversation_status',
        'ai_enabled',
        'last_message_at',
        'unread_count',
        'sla_deadline',
        'internal_note',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'dob' => 'date',
            'is_opted_out' => 'boolean',
            'opted_out' => 'boolean',
            'custom_values' => 'array',
            'last_messaged_at' => 'datetime',
            'last_marketed_at' => 'datetime',
            'last_message_at' => 'datetime',
            'sla_deadline' => 'datetime',
            'ai_enabled' => 'boolean',
        ];
    }

    /**
     * Get the groups that belong to the contact.
     */
    public function groups()
    {
        return $this->belongsToMany(Group::class);
    }

    /**
     * Get all of the contact's tags.
     */
    public function tags()
    {
        return $this->morphToMany(Tag::class, 'taggable');
    }

    /**
     * Get all messages for the contact.
     */
    public function messages()
    {
        return $this->hasMany(Message::class);
    }

    /**
     * Get the latest message for the contact.
     */
    public function latestMessage()
    {
        return $this->hasOne(Message::class)->latestOfMany();
    }

    public function notes()
    {
        return $this->hasMany(ContactNote::class)->orderByDesc('pinned')->orderByDesc('created_at');
    }

    public function assignee()
    {
        return $this->belongsTo(\App\Models\User::class, 'assigned_to');
    }
}
