<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;

class User extends Authenticatable
{
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasFactory, Notifiable;

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        'organization_id',
        'name',
        'email',
        'password',
        'role',
        'status',
        'two_factor_enabled',
        'two_factor_secret',
        'organization_role_id',
        'column_preferences',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
        'two_factor_secret',
        'organization_role_id',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
            'two_factor_enabled' => 'boolean',
            'column_preferences' => 'array',
        ];
    }

    /**
     * Get the organization that the user belongs to.
     */
    public function organizationRole() { return $this->belongsTo(OrganizationRole::class); }

    public function organization(): BelongsTo
    {
        return $this->belongsTo(Organization::class);
    }

    /**
     * Normalized (lowercase) role key used for comparisons.
     */
    public function roleKey(): string
    {
        return strtolower($this->role ?? '');
    }

    /**
     * Check if the user has any of the provided roles.
     *
     * @param  array<int, string>|string  $roles
     */
    public function hasRole(array|string $roles): bool
    {
        $roles = is_array($roles) ? $roles : func_get_args();

        return in_array($this->roleKey(), array_map('strtolower', $roles), true);
    }

    /**
     * Determine if the current user can manage the tenant team (Owner/System Admin).
     */
    public function canManageTeam(): bool
    {
        return $this->hasRole(['owner', 'system_admin']);
    }

    public function canManageCustomFields(): bool
    {
        if (!$this->organization) {
            return false;
        }
        return $this->hasRole(['owner', 'system_admin']);
    }

    /**
     * Check if the user is a super admin.
     */
    public function isSuperAdmin(): bool
    {
        return $this->roleKey() === 'system_admin';
    }
}
