<?php

namespace App\Services\Ai\Drivers;

use App\Services\Ai\Contracts\AiDriverInterface;
use Illuminate\Support\Facades\Http;

class AnthropicDriver implements AiDriverInterface
{
    protected string $apiKey;
    protected string $model;
    protected string $baseUrl = 'https://api.anthropic.com/v1/messages';
    protected string $apiVersion = '2023-06-01';

    public function __construct(string $apiKey, string $model = 'claude-3-opus-20240229')
    {
        $this->apiKey = $apiKey;
        $this->model = $model;
    }

    public function chat(array $messages, array $options = []): string
    {
        // Anthropic requires alternating user/assistant roles.
        // If there's a system message, we extract it.
        $systemPrompt = '';
        $formattedMessages = [];

        foreach ($messages as $msg) {
            if ($msg['role'] === 'system') {
                $systemPrompt .= $msg['content'] . "\n";
            } else {
                $formattedMessages[] = [
                    'role' => $msg['role'] === 'assistant' ? 'assistant' : 'user',
                    'content' => $msg['content'],
                ];
            }
        }

        $payload = [
            'model' => $options['model'] ?? $this->model,
            'messages' => $formattedMessages,
            'max_tokens' => $options['max_tokens'] ?? 1024,
            'temperature' => $options['temperature'] ?? 0.7,
        ];

        if (!empty($systemPrompt)) {
            $payload['system'] = trim($systemPrompt);
        }

        $response = Http::withHeaders([
            'x-api-key' => $this->apiKey,
            'anthropic-version' => $this->apiVersion,
            'content-type' => 'application/json',
        ])->post($this->baseUrl, $payload);

        if ($response->failed()) {
            throw new \Exception('Anthropic API error: ' . $response->body());
        }

        return $response->json('content.0.text') ?? '';
    }
}
