<?php

namespace App\Services;

use App\Models\MetaError;
use App\Models\Organization;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class MetaApiService
{
    protected string $baseUrl;
    protected string $version;
    protected ?Organization $organization = null;

    public function __construct()
    {
        $this->baseUrl = config('meta.url');
        $this->version = config('meta.version');
    }

    /**
     * Set the organization context for the API calls.
     */
    public function setOrganization(Organization $organization): self
    {
        $this->organization = $organization;
        return $this;
    }

    /**
     * Build the full API URL for a given endpoint.
     */
    public function buildUrl(string $endpoint): string
    {
        return sprintf('%s/%s/%s', rtrim($this->baseUrl, '/'), $this->version, ltrim($endpoint, '/'));
    }

    /**
     * Send a GET request to the Meta API.
     */
    public function get(string $endpoint, array $query = [])
    {
        return $this->request('get', $endpoint, $query);
    }

    /**
     * Send a GET request to the Meta API without any organization context.
     *
     * Useful for OAuth and debug endpoints where a bearer token is not yet available.
     */
    public function publicGet(string $endpoint, array $query = [])
    {
        $url = $this->buildUrl($endpoint);

        $response = Http::get($url, $query);

        if ($response->failed()) {
            Log::error("Meta API Public Error: {$endpoint}", [
                'status' => $response->status(),
                'payload' => $response->json(),
            ]);
            $response->throw();
        }

        return $response->json();
    }

    public function tokenGet(string $token, string $endpoint, array $query = [])
    {
        $url = $this->buildUrl($endpoint);

        $response = Http::withToken($token)->get($url, $query);

        if ($response->failed()) {
            Log::error("Meta API Token GET Error: {$endpoint}", [
                'status' => $response->status(),
                'payload' => $response->json(),
            ]);
            $response->throw();
        }

        return $response->json();
    }

    public function tokenPost(string $token, string $endpoint, array $data = [])
    {
        $url = $this->buildUrl($endpoint);

        $response = Http::asForm()->withToken($token)->post($url, $data);

        if ($response->failed()) {
            Log::error("Meta API Token POST Error: {$endpoint}", [
                'status' => $response->status(),
                'payload' => $response->json(),
            ]);
            $response->throw();
        }

        return $response->json();
    }

    /**
     * Send a POST request to the Meta API.
     */
    public function post(string $endpoint, array $data = [])
    {
        return $this->request('post', $endpoint, $data);
    }

    /**
     * Send a DELETE request to the Meta API.
     */
    public function delete(string $endpoint, array $data = [])
    {
        return $this->request('delete', $endpoint, $data);
    }

    /**
     * Execute the HTTP request to the Meta API.
     */
    protected function request(string $method, string $endpoint, array $payload = [])
    {
        if (!$this->organization) {
            throw new \Exception('Organization context is required for Meta API calls.');
        }

        if (!$this->organization->system_user_token) {
            throw new \Exception('System user token is missing for the organization.');
        }

        $url = $this->buildUrl($endpoint);

        $response = Http::withToken($this->organization->system_user_token)
            ->$method($url, $payload);

        if ($response->failed()) {
            $this->logError($endpoint, $response->status(), $response->json());
            $response->throw();
        }

        return $response->json();
    }

    /**
     * Log Meta API errors to the database for monitoring.
     */
    protected function logError(string $endpoint, int $statusCode, ?array $errorPayload): void
    {
        if ($this->organization) {
            MetaError::create([
                'organization_id' => $this->organization->id,
                'endpoint' => $endpoint,
                'status_code' => $statusCode,
                'error_payload' => $errorPayload ?? [],
            ]);
        }

        Log::error("Meta API Error: {$endpoint}", [
            'status' => $statusCode,
            'payload' => $errorPayload,
            'organization_id' => $this->organization?->id,
        ]);
    }
}
