<?php

namespace App\Services;

use App\Models\PlatformSetting;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Database\QueryException;

class PlatformSettingsService
{
    public function get(string $key, mixed $default = null, bool $decrypt = true): mixed
    {
        $cacheKey = "platform_setting:{$key}";

        try {
            $setting = Cache::rememberForever($cacheKey, function () use ($key) {
                return PlatformSetting::query()->where('key', $key)->first();
            });
        } catch (QueryException $e) {
            return $default;
        }

        if (!$setting) {
            return $default;
        }

        if ($setting->encrypted && $decrypt && $setting->value !== null) {
            try {
                return Crypt::decryptString($setting->value);
            } catch (\Throwable $e) {
                return $default;
            }
        }

        return $setting->value ?? $default;
    }

    public function set(string $key, mixed $value, bool $encrypt = false): void
    {
        $cacheKey = "platform_setting:{$key}";

        $storedValue = $value;

        if ($value !== null && $encrypt) {
            $storedValue = Crypt::encryptString((string) $value);
        }

        try {
            PlatformSetting::query()->updateOrCreate(
                ['key' => $key],
                ['value' => $storedValue, 'encrypted' => $encrypt]
            );
        } catch (QueryException $e) {
            return;
        }

        Cache::forget($cacheKey);
    }

    public function has(string $key): bool
    {
        try {
            return PlatformSetting::query()->where('key', $key)->exists();
        } catch (QueryException $e) {
            return false;
        }
    }
}
