import { computed } from 'vue';

export function usePermissions(user) {
    // Get user permissions from their role
    const userPermissions = computed(() => {
        if (!user.value) return [];
        
        // Owner has all permissions
        if (user.value.role === 'Owner') {
            return [
                'manage_contacts', 'send_messages', 'manage_campaigns', 'manage_automations',
                'view_reports', 'inbox_access', 'reply_messages', 'manage_templates',
                'manage_team', 'manage_billing', 'manage_settings'
            ];
        }
        
        // System Admin has all permissions
        if (user.value.role === 'System_Admin') {
            return [
                'manage_contacts', 'send_messages', 'manage_campaigns', 'manage_automations',
                'view_reports', 'inbox_access', 'reply_messages', 'manage_templates',
                'manage_team', 'manage_billing', 'manage_settings'
            ];
        }
        
        // Get permissions from custom role
        if (user.value.organization_role && user.value.organization_role.permissions) {
            return user.value.organization_role.permissions;
        }
        
        // Default Agent permissions
        return ['inbox_access', 'reply_messages'];
    });
    
    // Check if user has specific permission
    const hasPermission = (permission) => {
        return userPermissions.value.includes(permission);
    };
    
    // Check if user has any of the specified permissions
    const hasAnyPermission = (permissions) => {
        return permissions.some(permission => userPermissions.value.includes(permission));
    };
    
    // Check if user has all specified permissions
    const hasAllPermissions = (permissions) => {
        return permissions.every(permission => userPermissions.value.includes(permission));
    };
    
    // Menu items with their required permissions
    const menuItems = [
        {
            name: 'Dashboard',
            href: '/dashboard',
            icon: 'HomeIcon',
            permissions: [], // Everyone can see dashboard
            category: 'main'
        },
        {
            name: 'Chats & Inbox',
            href: '/chat',
            icon: 'ChatBubbleLeftRightIcon',
            permissions: ['inbox_access'],
            category: 'communication'
        },
        {
            name: 'Contacts & Groups',
            href: '/contacts',
            icon: 'UsersIcon',
            permissions: ['manage_contacts'],
            category: 'communication'
        },
        {
            name: 'Manage Campaigns',
            href: '/campaigns',
            icon: 'BullhornIcon',
            permissions: ['manage_campaigns'],
            category: 'communication'
        },
        {
            name: 'Message Templates',
            href: '/templates',
            icon: 'DocumentTextIcon',
            permissions: ['manage_templates'],
            category: 'communication'
        },
        {
            name: 'Automations & AI',
            href: '/automations',
            icon: 'CpuChipIcon',
            permissions: ['manage_automations'],
            category: 'automation'
        },
        {
            name: 'Manage Team',
            href: '/team',
            icon: 'UserGroupIcon',
            permissions: ['manage_team'],
            category: 'organization'
        },
        {
            name: 'Billing & Subscriptions',
            href: '/billing',
            icon: 'CreditCardIcon',
            permissions: ['manage_billing'],
            category: 'organization'
        },
        {
            name: 'Settings',
            href: '/settings',
            icon: 'Cog6ToothIcon',
            permissions: ['manage_settings'],
            category: 'system'
        },
        {
            name: 'Support',
            href: '/support',
            icon: 'LifebuoyIcon',
            permissions: [], // Everyone can see support
            category: 'system'
        },
        {
            name: 'Developer Tools',
            href: '/developer',
            icon: 'CodeBracketIcon',
            permissions: ['manage_settings'], // Only admins can see developer tools
            category: 'system'
        }
    ];
    
    // Filter menu items based on user permissions
    const filteredMenuItems = computed(() => {
        return menuItems.filter(item => {
            // If no permissions required, show to everyone
            if (item.permissions.length === 0) return true;
            
            // Check if user has required permissions
            return hasAnyPermission(item.permissions);
        });
    });
    
    // Group menu items by category
    const menuCategories = computed(() => {
        const categories = {
            main: { title: 'Main', items: [] },
            communication: { title: 'Communication', items: [] },
            automation: { title: 'Automation', items: [] },
            organization: { title: 'Organization', items: [] },
            system: { title: 'System', items: [] }
        };
        
        filteredMenuItems.value.forEach(item => {
            if (categories[item.category]) {
                categories[item.category].items.push(item);
            }
        });
        
        return categories;
    });
    
    return {
        userPermissions,
        hasPermission,
        hasAnyPermission,
        hasAllPermissions,
        filteredMenuItems,
        menuCategories
    };
}
